package vital_ecg;

import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;

import data.ECGConst;
import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.scene.chart.Axis;
import javafx.scene.chart.NumberAxis;
import javafx.scene.chart.XYChart;

/**
 * Prepravka pro uchovani dat potrebne pro grafy.
 *
 * @author Patrik Patera
 * @version 1.00
 */
public class ECGData {

    /**
     * Liste vsech vytvorenych grafu.
     */
    private final ObservableList<XYChart> allECGCharts;

    /**
     * X-ova osa ECG1 grafu.
     */
    private final Axis xAxisECG = new NumberAxis(ECGConst.XAXIS_MIN, ECGConst.XAXIS_MAX, 1);
    /**
     * Y-ova osa ECG1 grafu.
     */
    private final Axis yAxisECG = new NumberAxis(-0.5, 1, 1);

    /**
     * X-ova osa ECGX grafu.
     */
    private final Axis xAxisECG2 = new NumberAxis(ECGConst.XAXIS_MIN, ECGConst.XAXIS_MAX, 1);
    /**
     * Y-ova osa ECGX grafu.
     */
    private final Axis yAxisECG2 = new NumberAxis(-1, 0.5, 1);

    /**
     * X-ova osa CO2 grafu.
     */
    private final NumberAxis xAxisCO2 = new NumberAxis(ECGConst.XAXIS_MIN, ECGConst.XAXIS_MAX, 1);
    /**
     * Y-ova osa CO2 grafu.
     */
    private final NumberAxis yAxisCO2 = new NumberAxis(0, 60, 30);

    /**
     * Reference na hlavni graf s ECG1 (EKG) krivkou.
     */
    private final ECGLineChart graphECG1;
    /**
     * Reference na graf s CO2.
     */
    private final CO2Chart graphCO2;
    /**
     * Reference na vedlejsi graf s ECGX krivkou.
     */
  // private final ECGLineChart graphECG2;

    /**
     * Manager pro spusteni vlakna vykreslovani.
     */
    private final ScheduledExecutorService executor;

    /**
     * Fronta pridavanych casu pro jejich plynule vykreslovani.
     */
    private final Queue<ECGPoint> timeQueue;

    /**
     * Trida zarizujici animaci krivek.
     */
    private final ECGAnimation animation;

    /**
     * Trida zarizujici vypocet bodu pro danou krivku.
     */
    private final ECGCalculation calculation;

    /**
     * Indikuje pocet zatim nevykreslenych bodu.
     */
    private int countPoints;

    /**
     * Konstruktor zalozi list pro vechny a postupne vytvori vsechny grafy.
     */
    public ECGData() {
        allECGCharts = FXCollections.observableArrayList();
        executor = Executors.newScheduledThreadPool(2);
        timeQueue = new ConcurrentLinkedQueue<>();

        graphECG1 = setECGGraph();
        //graphECG2 = setECG2Graph(); //Pro ECG2 nejsou korektni data
        graphCO2 = setCO2Graph();

        animation = new ECGAnimation(executor, this);
        calculation = new ECGCalculation(executor, this);
        calculation.setNewCalculation();
    }

    /**
     * Pocet zatim nevykreslenych bodu.
     *
     * @return pocet zatim nevykreslenych bodu.
     */
    public int getCountPoints() {
        return countPoints;
    }

    /**
     * Pri vytvoreni noveho bodu se inkrementuje jejich celkovy pocet.
     */
    synchronized public void incPoint() {
        countPoints++;
    }

    /**
     * Pri vytvoreni noveho bodu se dekrementuje jejich pocet.
     */
    synchronized public void decPoint() {
        countPoints--;
    }

    /**
     * Poslani / pridani nove krivky z ovladace.
     *
     * @param data list krivek, ktere obsahuji jednotlive vzorky.
     */
    public void sendCurve(ObservableList<ObservableList<ECGPoint>> data) {
        if(data == null){
            return;
        }

        calculation.setECGData(data);
        
        graphECG1.setTypeCurve(false);
       // graphECG2.setTypeCurve(false);
    }

    /**
     * Nastaveni srdecniho rytmu.
     *
     * @param hr hodnota srdecniho rytmu.
     */
    public void setHR(int hr) {
        if(hr > 0){
            calculation.setHR(hr);
        }
    }

    /**
     * Pridani bodu do fronty bodu.
     *
     * @param data bod pridavany do fronty.
     */
    public void sendData(ECGPoint data) {
        timeQueue.add(data);
    }

    /**
     * Odstraneni bodu z fronty a prirazeni pro kresleni.
     *
     * @return bod z fronty bodu.
     */
    public ECGPoint removePoint() {
        return timeQueue.remove();
    }

    /**
     * Zjistuje, jestli je list prazdny.
     *
     * @return true - list prazdny.
     */
    public boolean isListEmpty() {
        return timeQueue.isEmpty();
    }

    /**
     * Nastaveni vykreslovani krivky, kdy monitor neni pripojen.
     */
    public void notConnected() {
        if (calculation != null) {
            calculation.notConnected();
        }

        graphECG1.setTypeCurve(true);
       // graphECG2.setTypeCurve(true);
    }

    /**
     * Nastaveni vykreslovani asystolie krivky.
     */
    public void plotAsystolie() {
        /* Stejne parametry jako pro notConnected, ale krivka neni teckovana */
        this.notConnected();

        graphECG1.setTypeCurve(false);
       // graphECG2.setTypeCurve(false);
    }

    /**
     * Zmena dobu cekani pri vykreslovani bodu.
     *
     * @param time nova doba cekani.
     */
    public void changeWait(double time) {
        if (animation != null && calculation != null) {
            animation.getAnimation().cancel(true);
            calculation.getCalculation().cancel(true);
        }

        calculation.setNewCalculation();
        animation.setNewAnimation(time);

    }

    /**
     * Metoda, ktera zajistuje spravne nastaveni casu vykreslovani jednotlivych
     * bodu.
     *
     * @param time cas, ktery se ma nastavit.
     */
    public void setWait(double time) {
        animation.setNewAnimation(time);
    }

    /**
     * Vytvoreni a nastaveni CO2 grafu.
     *
     * @return CO2 graf
     */
    private CO2Chart setCO2Graph() {
        System.out.println("Creating new chart: ");
        CO2Chart chart = new CO2Chart(xAxisCO2, yAxisCO2);
        allECGCharts.add(chart);
        System.out.println("Chart - CO2 was created.");

        return chart;
    }

    /**
     * Vytvoreni grafu s urcitou bodovou radou.
     *
     * @return spojity graf.
     */
    public ECGLineChart setECGGraph() {
        System.out.println("Creating new chart: ");

        ECGLineChart chart = new ECGLineChart(xAxisECG, yAxisECG);

        chart.setCreateSymbols(false);
        chart.setLegendVisible(false);

        allECGCharts.add(chart);
        System.out.println("Chart - ECG1 was created.");

        return chart;
    }

    /**
     * Vytvoreni grafu s urcitou bodovou radou.
     *
     * @return spojity graf.
     */
    public ECGLineChart setECG2Graph() {
        System.out.println("Creating new chart: ");

        ECGLineChart chart = new ECGLineChart(xAxisECG2, yAxisECG2);

        chart.setCreateSymbols(false);
        chart.setLegendVisible(false);

        allECGCharts.add(chart);
        System.out.println("Chart - ECG2 was created.");

        return chart;
    }

    /**
     * Vrati seznam dostupnych (vytvorenych) grafu.
     *
     * @return seznam vsech grafu.
     */
    public ObservableList<XYChart> getAllECGCharts() {
        return this.allECGCharts;
    }

    /**
     * Vrati referenci na gref, ktery zobrazuje ECG1.
     *
     * @return graf ECG1.
     */
    public ECGLineChart getGraphECG1() {
        return this.graphECG1;
    }

    /**
     * Vrati referenci na gref, ktery zobrazuje ECG2.
     *
     * @return graf ECG2.
     */
    public ECGLineChart getGraphECG2() {
      //  return this.graphECG2;
        return null;
    }

    /**
     * Vrati refernci na graf, ktery zobrazuje CO2.
     *
     * @return CO2 graf.
     */
    public CO2Chart getGraphCO2() {
        return this.graphCO2;
    }

    /**
     * Spravne ukonceni vlaken pro vypocet bodu a animaci.
     */
    public void clear() {
        graphCO2.die();
        executor.shutdownNow();
    }
    
    public void setBeep(boolean beep) {
        animation.setBeep(beep);
    }

    public void setAnimation(boolean flag) {
        graphCO2.setHasAnimation(flag);
        animation.hasAnimation(flag);
    }
}
